-- ============================================================= --
-- FOG CONTROL MOD
-- Autor: BMprod
-- date: 20.05.2025
-- dateUpdate: 28.07.2025
-- ============================================================= --
FogControl = {}
FogControl.name = g_currentModName
FogControl.path = g_currentModDirectory

FogControl.SETTINGS = {}
FogControl.CONTROLS = {}

addModEventListener(FogControl)

FogControl.menuItems = {
    'groundFogDensity',
    'heightFogDensity',
    'fogCoverage',
    'groundFogExtraHeight',
    'heightFogMaxHeight',
}

-- Default settings
FogControl.groundFogDensity = 0.01
FogControl.heightFogDensity = 0.1
FogControl.fogCoverage = 0.01
FogControl.groundFogExtraHeight = 7
FogControl.heightFogMaxHeight = 550
FogControl.fogSeasons = {
    [Season.SPRING] = true,
    [Season.SUMMER] = true,
    [Season.AUTUMN] = true,
}

-- Helper function to get table keys
function table.keys(tbl)
    local keys = {}
    for key, _ in pairs(tbl or {}) do
        table.insert(keys, tostring(key))
    end
    return keys
end

function FogControl:updateFogSettings(forceUpdate)
    if not g_currentMission or not g_currentMission.environment or not g_currentMission.environment.weather then
        print("FogControl: Cannot update fog settings - mission or environment not available")
        return
    end

    local fogUpdater = g_currentMission.environment.weather.fogUpdater
    local fogSettings = FogSettings.new()

    -- Apply settings from FogControl
    fogSettings.groundFogGroundLevelDensity = FogControl.groundFogDensity
    fogSettings.heightFogGroundLevelDensity = FogControl.heightFogDensity
    fogSettings.groundFogCoverageEdge0 = FogControl.fogCoverage
    fogSettings.groundFogCoverageEdge1 = math.max(FogControl.fogCoverage + 0.1, 1.0)
    fogSettings.groundFogExtraHeight = FogControl.groundFogExtraHeight
    fogSettings.groundFogMinValleyDepth = 1.5
    fogSettings.heightFogMaxHeight = FogControl.heightFogMaxHeight
    fogSettings.groundFogStartDayTimeMinutes = 240
    fogSettings.groundFogEndDayTimeMinutes = 600
    fogSettings.groundFogWeatherTypes = {}

    -- Update fogUpdater state
    fogUpdater.visibilityAlpha = (FogControl.groundFogDensity > 0 or FogControl.heightFogDensity > 0) and 1 or 0
    fogUpdater.alpha = 0 -- Reset alpha to force immediate interpolation
    fogUpdater.duration = 30000 -- Set transition duration

    -- Force apply settings to currentFog
    fogUpdater.currentFog.groundFogGroundLevelDensity = FogControl.groundFogDensity
    fogUpdater.currentFog.heightFogGroundLevelDensity = FogControl.heightFogDensity
    fogUpdater.currentFog.groundFogCoverageEdge0 = FogControl.fogCoverage
    fogUpdater.currentFog.groundFogCoverageEdge1 = math.max(FogControl.fogCoverage + 0.1, 1.0)
    fogUpdater.currentFog.groundFogExtraHeight = FogControl.groundFogExtraHeight
    fogUpdater.currentFog.groundFogMinValleyDepth = 1.5
    fogUpdater.currentFog.heightFogMaxHeight = FogControl.heightFogMaxHeight
    fogUpdater.currentFog.groundFogStartDayTimeMinutes = 240
    fogUpdater.currentFog.groundFogEndDayTimeMinutes = 600
    fogUpdater.currentFog.groundFogWeatherTypes = {}

    fogUpdater:setTargetFog(fogSettings, 30000)
    fogUpdater.isDirty = true

    -- Debug log to track applied values
    print(string.format("FogControl: Applied groundFogDensity=%.4f, heightFogDensity=%.4f, fogCoverage=%.4f, groundFogExtraHeight=%.2f, heightFogMaxHeight=%.2f, visibilityAlpha=%.2f",
        fogUpdater.currentFog.groundFogGroundLevelDensity,
        fogUpdater.currentFog.heightFogGroundLevelDensity,
        fogUpdater.currentFog.groundFogCoverageEdge0,
        fogUpdater.currentFog.groundFogExtraHeight,
        fogUpdater.currentFog.heightFogMaxHeight,
        fogUpdater.visibilityAlpha))

    if g_server ~= nil and not manualUpdate then
        local lastMieScale = fogUpdater.lastMieScale or 1.0
        local alpha = fogUpdater.alpha or 0.0
        local nightFactor = g_currentMission.environment.weather.nightFactor or 0.0
        local dayFactor = g_currentMission.environment.weather.dayFactor or 1.0
        g_server:broadcastEvent(FogStateEvent.new(
            fogSettings.groundFogGroundLevelDensity,
            lastMieScale,
            alpha,
            30000,
            nightFactor,
            dayFactor
        ))
    end
end

function FogControl:loadMap(name)
    FogControl:readSettings()
    addConsoleCommand("gsFogControl", "Control fog settings", "consoleCommandFogControl", FogControl)
    if g_currentMission ~= nil then
        FSBaseMission.registerToLoadOnMapFinished(g_currentMission, FogControl)
    end
    FogControl.initialised = true
    FogControl:updateFogSettings(true)
end

function FogControl:onLoadMapFinished()
    FogControl:readSettings()
    FogControl:updateFogSettings(true)
end

function FogControl:onDayChanged()
    FogControl:updateFogSettings(true)
end

function FogControl:consoleCommandFogControl(param1, param2)
    if param1 == "groundDensity" then
        local value = tonumber(param2) or FogControl.groundFogDensity
        FogControl.groundFogDensity = math.clamp(value, 0.0, 1.0)
        FogControl:writeSettings()
        FogControl:updateFogSettings(true)
        return string.format("Ground fog density: %.4f", FogControl.groundFogDensity)
    elseif param1 == "heightDensity" then
        local value = tonumber(param2) or FogControl.heightFogDensity
        FogControl.heightFogDensity = math.clamp(value, 0.0, 1.0)
        FogControl:writeSettings()
        FogControl:updateFogSettings(true)
        return string.format("Height fog density: %.4f", FogControl.heightFogDensity)
    elseif param1 == "coverage" then
        local value = tonumber(param2) or FogControl.fogCoverage
        FogControl.fogCoverage = math.clamp(value, 0.0, 1.0)
        FogControl:writeSettings()
        FogControl:updateFogSettings(true)
        return string.format("Fog coverage: %.4f", FogControl.fogCoverage)
    elseif param1 == "groundHeight" then
        local value = tonumber(param2) or FogControl.groundFogExtraHeight
        FogControl.groundFogExtraHeight = math.clamp(value, 0.0, 150.0)
        FogControl:writeSettings()
        FogControl:updateFogSettings(true)
        return string.format("Ground fog extra height: %.2f", FogControl.groundFogExtraHeight)
    elseif param1 == "heightMax" then
        local value = tonumber(param2) or FogControl.heightFogMaxHeight
        FogControl.heightFogMaxHeight = math.clamp(value, 0.0, 1500.0)
        FogControl:writeSettings()
        FogControl:updateFogSettings(true)
        return string.format("Height fog max height: %.2f", FogControl.heightFogMaxHeight)
    elseif param1 == "seasons" then
        local seasons = string.split(param2, ",")
        FogControl.fogSeasons = {}
        for _, seasonName in ipairs(seasons) do
            local season = Season.getByName(seasonName:upper())
            if season then
                FogControl.fogSeasons[season] = true
            end
        end
        FogControl:writeSettings()
        FogControl:updateFogSettings(true)
        return string.format("Fog seasons: %s", table.concat(table.keys(FogControl.fogSeasons), ", "))
    end
    return "Usage: gsFogControl [groundDensity|heightDensity|coverage|groundHeight|heightMax|seasons] <value>"
end

function FogControl:setValue(id, value)
    FogControl[id] = value
    print(string.format("FogControl: Set %s = %.4f", id, value))
end

function FogControl:getValue(id)
    return FogControl[id]
end

function FogControl:getStateIndex(id, value)
    local value = value or FogControl:getValue(id)
    local setting = FogControl.SETTINGS[id]
    if not setting then
        return 1
    end
    local values = setting.values

    if type(value) == 'number' then
        local index = setting.default
        local initialDiff = math.huge
        for i, v in ipairs(values) do
            local currentDiff = math.abs(v - value)
            if currentDiff < initialDiff then
                initialDiff = currentDiff
                index = i
            end
        end
        return index
    else
        for i, v in ipairs(values) do
            if v == value then
                return i
            end
        end
        return setting.default
    end
end

function FogControl:writeSettings()
    local key = "fogControlSettings"
    local userSettingsFile = Utils.getFilename("modSettings/FogControl.xml", getUserProfileAppPath())
    local xmlFile = createXMLFile("settings", userSettingsFile, key)

    if xmlFile ~= 0 then
        for _, id in pairs(FogControl.menuItems) do
            local value = FogControl:getValue(id)
            local xmlValueKey = "fogControlSettings." .. id .. "#value"
            setXMLFloat(xmlFile, xmlValueKey, value)
            print(string.format("FogControl: Saved %s = %.4f to XML", id, value))
        end
        local seasonsValue = table.concat(table.keys(FogControl.fogSeasons), ",")
        setXMLString(xmlFile, "fogControlSettings.fogSeasons#value", seasonsValue)
        print(string.format("FogControl: Saved fogSeasons = %s to XML", seasonsValue))
        saveXMLFile(xmlFile)
        delete(xmlFile)
    else
        print("FogControl: Failed to create XML file for saving settings")
    end
end

function FogControl:readSettings()
    local userSettingsFile = Utils.getFilename("modSettings/FogControl.xml", getUserProfileAppPath())
    if not fileExists(userSettingsFile) then
        print("FogControl: Settings file not found, writing defaults")
        FogControl:writeSettings()
        return
    end

    local xmlFile = loadXMLFile("fogControlSettings", userSettingsFile)
    if xmlFile ~= 0 then
        for _, id in pairs(FogControl.menuItems) do
            local xmlValueKey = "fogControlSettings." .. id .. "#value"
            if hasXMLProperty(xmlFile, xmlValueKey) then
                local value = getXMLFloat(xmlFile, xmlValueKey)
                if value ~= nil then
                    FogControl:setValue(id, value)
                else
                    print(string.format("FogControl: Failed to load %s from XML, using default %.4f", id, FogControl:getValue(id)))
                end
            else
                print(string.format("FogControl: Property %s not found in XML, using default %.4f", id, FogControl:getValue(id)))
            end
        end
        local seasonsKey = "fogControlSettings.fogSeasons#value"
        if hasXMLProperty(xmlFile, seasonsKey) then
            local strValue = getXMLString(xmlFile, seasonsKey)
            FogControl.fogSeasons = {}
            if strValue then
                local items = string.split(strValue, ",")
                for _, item in ipairs(items) do
                    item = item:trim():upper()
                    local season = Season.getByName(item)
                    if season then
                        FogControl.fogSeasons[season] = true
                    end
                end
                print(string.format("FogControl: Loaded fogSeasons = %s", table.concat(table.keys(FogControl.fogSeasons), ",")))
            end
            if not next(FogControl.fogSeasons) then
                FogControl.fogSeasons = {
                    [Season.SPRING] = true,
                    [Season.SUMMER] = true,
                    [Season.AUTUMN] = true,
                }
                print("FogControl: No valid seasons loaded, using defaults: SPRING,SUMMER,AUTUMN")
            end
        else
            print("FogControl: fogSeasons not found in XML, using defaults")
        end
        delete(xmlFile)
    else
        print("FogControl: Failed to load XML file")
    end
end

-- Menu integration
local percentValues = {}
local percentStrings = {}
for i = 0, 1000 do
    percentValues[i + 1] = i / 1000
    percentStrings[i + 1] = string.format("%.1f%%", i / 10)
end

local groundHeightValues = {}
local groundHeightStrings = {}
for i = 0, 150 do
    groundHeightValues[i + 1] = i
    groundHeightStrings[i + 1] = string.format("%.1f", i)
end

local heightMaxValues = {}
local heightMaxStrings = {}
for i = 0, 1500, 10 do
    heightMaxValues[#heightMaxValues + 1] = i
    heightMaxStrings[#heightMaxStrings + 1] = string.format("%.1f", i)
end

FogControl.SETTINGS.groundFogDensity = {
    default = 15,
    values = percentValues,
    strings = percentStrings
}

FogControl.SETTINGS.heightFogDensity = {
    default = 200,
    values = percentValues,
    strings = percentStrings
}

FogControl.SETTINGS.fogCoverage = {
    default = 100,
    values = percentValues,
    strings = percentStrings
}

FogControl.SETTINGS.groundFogExtraHeight = {
    default = 8,
    values = groundHeightValues,
    strings = groundHeightStrings
}

FogControl.SETTINGS.heightFogMaxHeight = {
    default = 56,
    values = heightMaxValues,
    strings = heightMaxStrings
}

function FogControl:addMenuOption(id)
    local inGameMenu = g_gui.screenControllers[InGameMenu]
    local settingsPage = inGameMenu.pageSettings
    local settingsLayout = settingsPage.generalSettingsLayout

    local original = #FogControl.SETTINGS[id].values == 2 and settingsPage.checkWoodHarvesterAutoCutBox or settingsPage.multiVolumeVoiceBox
    local options = FogControl.SETTINGS[id].strings

    local menuOptionBox = original:clone(settingsLayout)
    menuOptionBox.id = id .. "box"

    local menuOption = menuOptionBox.elements[1]
    menuOption.id = id
    menuOption.target = FogControl
    menuOption:setCallback("onClickCallback", "onMenuOptionChanged")
    menuOption:setDisabled(false)

    local toolTip = menuOption.elements[1]
    toolTip:setText(g_i18n:getText("tooltip_fogcontrol_" .. id))

    local setting = menuOptionBox.elements[2]
    setting:setText(g_i18n:getText("setting_fogcontrol_" .. id))

    menuOption:setTexts({unpack(options)})
    menuOption:setState(FogControl:getStateIndex(id))

    FogControl.CONTROLS[id] = menuOption

    local function updateFocusIds(element)
        if not element then return end
        element.focusId = FocusManager:serveAutoFocusId()
        for _, child in pairs(element.elements) do
            updateFocusIds(child)
        end
    end
    updateFocusIds(menuOptionBox)
    table.insert(settingsPage.controlsList, menuOptionBox)
end

function FogControl:onMenuOptionChanged(state, menuOption)
    local id = menuOption.id
    local value = FogControl.SETTINGS[id].values[state]
    FogControl:setValue(id, value)
    FogControl:writeSettings()
    FogControl:updateFogSettings(true) -- Apply settings immediately
end

-- Add menu section and options
local inGameMenu = g_gui.screenControllers[InGameMenu]
local settingsPage = inGameMenu.pageSettings
local settingsLayout = settingsPage.generalSettingsLayout

local sectionTitle
for _, elem in ipairs(settingsLayout.elements) do
    if elem.name == "sectionHeader" then
        sectionTitle = elem:clone(settingsLayout)
        break
    end
end
if sectionTitle then
    sectionTitle:setText(g_i18n:getText("menu_FogControl_TITLE"))
else
    local title = TextElement.new()
    title:applyProfile("fs25_settingsSectionHeader", true)
    title:setText(g_i18n:getText("menu_FogControl_TITLE"))
    title.name = "sectionHeader"
    settingsLayout:addElement(title)
    sectionTitle = title
end

sectionTitle.focusId = FocusManager:serveAutoFocusId()
table.insert(settingsPage.controlsList, sectionTitle)
FogControl.CONTROLS[sectionTitle.name] = sectionTitle

for _, id in pairs(FogControl.menuItems) do
    FogControl:addMenuOption(id)
end
settingsLayout:invalidateLayout()

-- Handle focus for custom controls
FocusManager.setGui = Utils.appendedFunction(FocusManager.setGui, function(_, gui)
    if gui == "ingameMenuSettings" then
        for _, control in pairs(FogControl.CONTROLS) do
            if not control.focusId or not FocusManager.currentFocusData.idToElementMapping[control.focusId] then
                if not FocusManager:loadElementFromCustomValues(control, nil, nil, false, false) then
                    Logging.warning("Could not register control %s with the focus manager", control.id or control.name or control.focusId)
                end
            end
        end
        local settingsPage = g_gui.screenControllers[InGameMenu].pageSettings
        settingsPage.generalSettingsLayout:invalidateLayout()
    end
end)

InGameMenuSettingsFrame.onFrameOpen = Utils.appendedFunction(InGameMenuSettingsFrame.onFrameOpen, function()
    local isAdmin = g_currentMission:getIsServer() or g_currentMission.isMasterUser
    for _, id in pairs(FogControl.menuItems) do
        local menuOption = FogControl.CONTROLS[id]
        menuOption:setState(FogControl:getStateIndex(id))
        menuOption:setDisabled(not isAdmin)
    end
end)

-- Override Weather.randomizeFog to apply FogControl settings
Weather.randomizeFog = Utils.overwrittenFunction(Weather.randomizeFog, function(self, superFunc, duration)
    FogControl:updateFogSettings(true)
end)

-- Override FogUpdater.update to ensure immediate application of FogControl settings
FogUpdater.update = Utils.overwrittenFunction(FogUpdater.update, function(self, superFunc, dt)
    if not FogControl.fogSeasons[g_currentMission.environment.currentSeason] then
        self.visibilityAlpha = 0
        self.isDirty = true
        return
    end

    if self.alpha < 1 then
        local newAlpha = self.alpha + dt / self.duration
        self.alpha = math.min(newAlpha, 1)
        local alpha = self.alpha
        local currentFog = self.currentFog
        local lastFog = self.lastFog
        local targetFog = self.targetFog
        currentFog.groundFogCoverageEdge0 = MathUtil.lerp(lastFog.groundFogCoverageEdge0, FogControl.fogCoverage, alpha)
        currentFog.groundFogCoverageEdge1 = MathUtil.lerp(lastFog.groundFogCoverageEdge1, math.max(FogControl.fogCoverage + 0.1, 1.0), alpha)
        currentFog.groundFogExtraHeight = MathUtil.lerp(lastFog.groundFogExtraHeight, FogControl.groundFogExtraHeight, alpha)
        currentFog.groundFogGroundLevelDensity = MathUtil.lerp(lastFog.groundFogGroundLevelDensity, FogControl.groundFogDensity, alpha)
        currentFog.groundFogMinValleyDepth = MathUtil.lerp(lastFog.groundFogMinValleyDepth, targetFog.groundFogMinValleyDepth, alpha)
        currentFog.heightFogMaxHeight = MathUtil.lerp(lastFog.heightFogMaxHeight, FogControl.heightFogMaxHeight, alpha)
        currentFog.heightFogGroundLevelDensity = MathUtil.lerp(lastFog.heightFogGroundLevelDensity, FogControl.heightFogDensity, alpha)
        self.isDirty = true
    end

    self.visibilityAlpha = (FogControl.groundFogDensity > 0 or FogControl.heightFogDensity > 0) and 1 or 0

    if self.isDirty then
        local currentFog = self.currentFog
        local coverageEdge0 = MathUtil.lerp(1, currentFog.groundFogCoverageEdge0, self.visibilityAlpha)
        local coverageEdge1 = MathUtil.lerp(1, currentFog.groundFogCoverageEdge1, self.visibilityAlpha)
        setGroundFogGlobalCoverage(coverageEdge0, coverageEdge1)
        setGroundFogHeight(currentFog.groundFogExtraHeight)
        setGroundFogGroundLevelDensity(currentFog.groundFogGroundLevelDensity)
        setGroundFogMinimumValleyDepth(currentFog.groundFogMinValleyDepth)
        setHeightFogGroundLevelDensity(currentFog.heightFogGroundLevelDensity)
        setHeightFogMaxHeight(currentFog.heightFogMaxHeight)
        self.isDirty = false
    end
end)

-- Override FogUpdater.setTargetFog to enforce FogControl settings
FogUpdater.setTargetFog = Utils.overwrittenFunction(FogUpdater.setTargetFog, function(self, superFunc, fogSettings, duration)
    local customFog = FogSettings.new()
    customFog.groundFogGroundLevelDensity = FogControl.groundFogDensity
    customFog.heightFogGroundLevelDensity = FogControl.heightFogDensity
    customFog.groundFogCoverageEdge0 = FogControl.fogCoverage
    customFog.groundFogCoverageEdge1 = math.max(FogControl.fogCoverage + 0.1, 1.0)
    customFog.groundFogExtraHeight = FogControl.groundFogExtraHeight
    customFog.groundFogMinValleyDepth = 1.5
    customFog.heightFogMaxHeight = FogControl.heightFogMaxHeight
    customFog.groundFogStartDayTimeMinutes = 240
    customFog.groundFogEndDayTimeMinutes = 600
    customFog.groundFogWeatherTypes = {}
    self.alpha = 0
    self.duration = math.max(1, duration or 30000)
    self.lastFog = self.currentFog:clone()
    self.targetFog = customFog
    self.currentFog.groundFogWeatherTypes = {}
    self.isDirty = true
end)