--[[
    MapZoomEvent

    Multiplayer Sync Event for the RealGPS's Map Zoom change.

	@author: 		BayernGamers
	@date: 			09.04.2025
	@version:		2.1

	History:		v1.0 @15.07.2023 - initial implementation in FS 22
                    ------------------------------------------------------------------------------------------------------
                    v2.0 @11.03.2025 - convert and re-write for FS25
                    ------------------------------------------------------------------------------------------------------
                    v2.1 @09.04.2025 - added playerSize to sync
                    ------------------------------------------------------------------------------------------------------
                    v2.2 @15.06.2025 - added support for multiple PDA's
                    -------------------------------------------------------------------------------------------------------
	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
MapZoomEvent = {}

local MapZoomEvent_mt = Class(MapZoomEvent, Event)

InitEventClass(MapZoomEvent, "MapZoomEvent")

function MapZoomEvent:emptyNew()
    local self = Event.new(MapZoomEvent_mt)
    return self
end

function MapZoomEvent.new(vehicle, mapId, zoom, playerSize)
    local self = MapZoomEvent:emptyNew()
    self.vehicle = vehicle
    self.mapId = mapId
    self.zoom = zoom
    self.playerSize = playerSize

    return self
end

function MapZoomEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.vehicle)
    streamWriteInt8(streamId, self.mapId)
    streamWriteFloat32(streamId, self.zoom)
    streamWriteFloat32(streamId, self.playerSize)
end

function MapZoomEvent:readStream(streamId, connection)
    self.vehicle = NetworkUtil.readNodeObject(streamId)
    self.mapId = streamReadInt8(streamId)
    self.zoom = streamReadFloat32(streamId)
    self.playerSize = streamReadFloat32(streamId)
    
    self:run(connection)
end

function MapZoomEvent:run(connection)
    if not connection:getIsServer() then
        g_server:broadcastEvent(self, false, connection, self.vehicle)
    end
    if self.vehicle ~= nil then
        for i, map in ipairs(self.vehicle.spec_realGPS.maps) do
            if i == self.mapId then
                self.vehicle:setMapZoom(map, self.zoom)
                self.vehicle:setPlayerSize(map, self.playerSize)
                RealGPS.updateActionEvents(self.vehicle)
            end
        end
    end
end

function MapZoomEvent.sendEvent(vehicle, mapId, zoom, playerSize, noEventSend)
    if noEventSend == nil or noEventSend == false then
        if g_server ~= nil then
            g_server:broadcastEvent(MapZoomEvent.new(vehicle, mapId, zoom, playerSize), nil, nil, vehicle)
        else
            g_client:getServerConnection():sendEvent(MapZoomEvent.new(vehicle, mapId, zoom, playerSize))
        end
    end
end